/*

  SmartClient Ajax RIA system
  Version SNAPSHOT_v15.0d_2026-02-03/LGPL Deployment (2026-02-03)

  Copyright 2000 and beyond Isomorphic Software, Inc. All rights reserved.
  "SmartClient" is a trademark of Isomorphic Software, Inc.

  LICENSE NOTICE
     INSTALLATION OR USE OF THIS SOFTWARE INDICATES YOUR ACCEPTANCE OF
     ISOMORPHIC SOFTWARE LICENSE TERMS. If you have received this file
     without an accompanying Isomorphic Software license file, please
     contact licensing@isomorphic.com for details. Unauthorized copying and
     use of this software is a violation of international copyright law.

  DEVELOPMENT ONLY - DO NOT DEPLOY
     This software is provided for evaluation, training, and development
     purposes only. It may include supplementary components that are not
     licensed for deployment. The separate DEPLOY package for this release
     contains SmartClient components that are licensed for deployment.

  PROPRIETARY & PROTECTED MATERIAL
     This software contains proprietary materials that are protected by
     contract and intellectual property law. You are expressly prohibited
     from attempting to reverse engineer this software or modify this
     software for human readability.

  CONTACT ISOMORPHIC
     For more information regarding license rights and restrictions, or to
     report possible license violations, please contact Isomorphic Software
     by email (licensing@isomorphic.com) or web (www.isomorphic.com).

*/
//> @class StyleSheetHandler
// This class allows developers to create, modify and load/unload custom CSS stylesheets with
// simple API calls.
// @treeLocation Client Reference/Style Editors
// @visibility external
//<

isc.defineClass("StyleSheetHandler", "Class");
isc.StyleSheetHandler.addClassProperties({
    _styleSheets: [],

    getSheet : function (name) {
        if (isc.isAn.Object(name)) return name;
        // return a stylesheet
        return this._styleSheets[name];
    }

});

isc.StyleSheetHandler.addProperties({
    //> @attr StyleSheetHandler.name (String : null : IR)
    // The name for this handler.  This value is applied as both the 'id' and 'title'
    // attributes on the handler's +link{styleSheetHandler.styleSheet, stylesheet}.
    // @visibility external
    //<
    name: null,

    //> @attr StyleSheetHandler.styleSheet (DOMElement : null : R)
    // A reference to this handler's stylesheet-element in the DOM.  This attribute cannot be 
    // set - see +link{StyleSheetHandler.injectSheet, injectSheet()} and 
    // +link{StyleSheetHandler.unload, unload()} for details.
    // @visibility external
    //<
    styleSheet: null,

    //> @attr StyleSheetHandler.cssText (String : null : IRW)
    // A string representation of all the CSS styles in this handler's 
    // +link{styleSheetHandler.styleSheet}.
    // @visibility external
    //<
    cssText: null,
    getCssText : function () {
        return this.filterCssText(this.getClassList());
    },
    //> @method StyleSheetHandler.setCssText()
    // Replace the entire CSS in this handler's +link{styleSheetHandler.styleSheet, stylesheet}.
    // <P>
    // If the stylesheet is already in the DOM, it is first 
    // +link{styleSheetHandler.unload, unloaded}, the +link{styleSheetHandler.cssText, cssText}
    // is updated and then the stylesheet is +link{styleSheetHandler.injectSheet, injected}
    // back into the DOM.  Otherwise, the handler's cssText is updated and no further action is 
    // taken.  
    // @param cssText (String) complete cssText to apply to the stylesheet
    // @visibility external
    //<
    setCssText : function (cssText) {
        if (this.loaded()) {
            // clear the classList cache - will be recalc'd by getClassList()
            this.clearClassListCache();
            // unload, set cssText, re-inject
            this.unload();
            this.cssText = cssText;
            this.injectSheet();
        } else {
            this.cssText = cssText;
        }
    },

    //> @attr StyleSheetHandler.autoLoad (boolean : false : IRW)
    // When true, this handler will automatically +link{styleSheetHandler.injectSheet, inject} 
    // it's stylesheet into the DOM.
    // @visibility external
    //<
    autoLoad: false,

    init : function () {
        var result = this.Super("init", arguments);
        if (this.autoLoad) this.injectSheet();
        return result;
    },

    //> @method StyleSheetHandler.injectSheet()
    // Create this handler's +link{StyleSheetHandler.styleSheet, stylesheet} element in the DOM.
    // @visibility external
    //<
    injectSheet : function (cssText) {
        if (this.styleSheet) return;
        if (cssText) this.cssText = cssText;
        this.createStyleSheet();
    },

    loadSkinStyleSheet : function (editorStyleSheet) {
        if (this.styleSheet) return;
        var sheets = document.styleSheets
        for (var i=0; i<sheets.length; i++) {
            var ss = sheets[i];
            var sheetName = editorStyleSheet ? "skin_styles_editor" : "skin_styles";
            if (ss.href && ss.href.contains(sheetName)) {
                this.sheetHref = ss.href;
                this.styleSheet = ss;
            }
        }
    },

    //> @method StyleSheetHandler.unload()
    // Remove this handler's +link{styleSheetHandler.styleSheet, stylesheet} from the DOM and 
    // destroy it.
    // @visibility external
    //<
    unload : function () {
        // if sheetHref is set, we loaded the skin_styles sheet, for the skin 
        if (this.sheetHref != null) {
            // remove a stylesheet from the document
            var parentNode = this.styleSheet.ownerNode.parentElement;
            parentNode.removeChild(this.styleSheet.ownerNode);
        }
        this.styleSheet = null;
        this.sheetHref = null;
    },

    createStyleSheet : function () {
        // create a stylesheet with rules created from this handler's cssText
        var el = document.createElement('style');

        // Append style element to head
        document.head.appendChild(el);

        // use id and title to find later for unload, eg
        el.sheet.id = this.name;
        el.sheet.title = this.name;
        
        this.styleSheet = el.sheet;

        if (this.cssText) this.injectCssText(this.cssText);
    },

    //> @method StyleSheetHandler.injectCssText()
    // Injects text defining a block of selectors into this handler's 
    // +link{styleSheetHandler.styleSheet, stylesheet}.
    // <P>
    // The passed 'cssText' string should contain one or more selectors, each in the format 
    // '.className [, className ...] { css settings; }'.  Use 
    // +link{styleSheetHandler.modifyClass, modifyClass(className, cssText-without-braces)} 
    // to apply styles to a specific CSS class.
    // @param cssText (String) string of css selectors and their settings to apply to this 
    //                        handler's +link{styleSheetHandler.styleSheet, stylesheet}.
    // @return (boolean) returns true if CSS was injected, false otherwise
    // @visibility external
    //<
    // @param [index] (String) insertion-index in the cssRules array.
    removeComments : function (input) {
        // Remove single line comments
        input = input.replace(/\/\/.*?\n/g, '');
        // Remove multi-line comments
        input = input.replace(/\/\*[\s\S]*?\*\//g, '');
        return input;
    },
    injectCssText : function (cssText, index) {
        this._injectingBlock = true;
        
        // remove comments from the CSS
        cssText = this.removeComments(cssText);
        
        // get an array of strings like ".className {content", without the trailing "}"
        var blocks = cssText.split("}").callMethod("trim"),
            modified = false
        ;
        for (var i=0; i<blocks.length; i++) {
            if (!blocks[i]) continue;
            var parts = blocks[i].split("{").callMethod("trim");
            // get the className (without leading period)
            var className = this.formatClassName(parts[0]);
            // get the css text
            var cssText = parts[1];
            // add a new rule that applies cssText to className
            if (this.modifyClass(className, cssText, index != null ? index++ : null)) {
                modified = true;
            }
        }
        delete this._injectingBlock;

        // clear the classList cache
        this.clearClassListCache();

        // see comment in this method
        this.refreshFramework();

        return modified;
    },
    
    //> @method StyleSheetHandler.modifyClass()
    // Inject a string of css settings into a single CSS class, via a new rule in this handler's 
    // +link{styleSheetHandler.styleSheet, stylesheet}.
    // <P>
    // The passed 'cssText' should contain semi-colon -separated CSS settings only, not the
    // selector/className or enclosing braces - { }.
    // @param className (String) name of the CSS class to apply the cssText to
    // @param cssText (String) string of css settings to apply to the passed className in this 
    //                        handler's +link{styleSheetHandler.styleSheet, stylesheet}.
    // @return (boolean) returns true if CSS was applied, false otherwise
    // @visibility external
    //<
    modifyClass : function (className, cssText, index) {
        // adds a new rule that applies "cssText" to "className"
        if (!className || !cssText) return false;
        if (isc.isAn.Object(cssText)) cssText = isc.JSON.encode(cssText);

        className = this.formatClassName(className);
        
        // @font-face rules should not have been prefixed with a ".", but strip it if it's there
        if (className.startsWith(".@")) className = className.substring(1);

        // Add the CSS to the stylesheet
        var sheet = this.styleSheet;
        if (index == null) index = sheet.cssRules.length;
        if ("insertRule" in sheet) {
            if (className) sheet.insertRule(className + "{" + cssText + "}", index);
            else sheet.insertRule(cssText, index);
        } else if ("addRule" in sheet) {
            sheet.addRule(className, cssText, index);
        }

        // see comment in this method
        if (!this._injectingBlock) this.refreshFramework();

        // delete the cache of classNames
        this.clearClassListCache();;
        return true;
    },

    //> @method StyleSheetHandler.removeClass()
    // Remove the passed CSS class from the +link{styleSheetHandler.styleSheet, stylesheet}, by
    // deleting it's rules and pruning it from multi-class selectors.
    // @param className (String) name of the CSS class to remove from the stylesheet
    // @return (boolean) returns true if CSS was removed, false otherwise
    // @visibility external
    //<
    removeClass : function (className) {
        var rules = isc.Canvas.getStyleRules(this.styleSheet);
        if (rules == null) return false;

        // make sure there's a leading "."
        var name = this.formatClassName(className);

        var modified = false;

        for (var i = rules.length - 1; i >= 0; i--) {
            var selectorText = rules[i].selectorText.trim();
            if (selectorText == null) continue;
            if (selectorText == name) {
                // rule is just for the passed className - remove it
                this.styleSheet.deleteRule(i);
                modified = true;
                continue;
            }
            // split the selectorText into classNames
            var classes = selectorText.split(",").callMethod("trim");
            var filtered = classes.filter(function (el) { return el != name; });
            if (classes.length != filtered.length) {
                // rule is for multiple classes, including the passed "className" - remove it
                // from the rule's selectorText
                rules[i].selectorText = filtered.join(", ");
                modified = true;
            }
        }
        return modified;
    },

    //> @method StyleSheetHandler.renameClass()
    // Rename the passed <i>oldClass</i> to <i>newClass</i> in all rules that reference it.
    // @param oldName (String) name of the CSS class to rename in the stylesheet
    // @param newName (String) new name for the CSS class
    // @return (boolean) returns true if a class was renamed, false otherwise
    // @visibility external
    //<
    renameClass : function (oldName, newName) {
        var rules = isc.Canvas.getStyleRules(this.styleSheet);
        if (rules == null) return false;

        // make sure there's a leading "."
        oldName = this.formatClassName(oldName);
        newName = this.formatClassName(newName);

        // bail if the passed className isn't in the stylesheet
        if (!this.hasClass(oldName)) return false;

        var modified = false;

        for (var i = rules.length - 1; i >= 0; i--) {
            var selectorText = rules[i].selectorText.trim();
            if (selectorText == null) continue;
            modified = true;
            // straight replace
            rules[i].selectorText = rules[i].selectorText.replaceAll(oldName, newName);
            //var regex = new RegExp("/\." + oldName + "\b/", "g");
            //rules[i].selectorText = rules[i].selectorText.replace(regex, newName);
        }
    },

    //> @method StyleSheetHandler.filterCssText()
    // Returns all styles for the passed className or array of classNames.  The output 
    // is a single compound string in the format 
    // "<i>.class1 { css settings; } .class2 { css settings; } ... </i>".
    // <P>
    // This string may then be imported directly into another stylesheet, either 
    // +link{styleSheetHandler.setCssText, replacing content} or 
    // +link{styleSheetHandler.injectCssText, adding to it}.
    // @param className (String | Array of String) one or more CSS-class names to get the 
    //                                             declaration for
    // @return (String) the CSS declaration for the passed className
    // @visibility external
    //<
    filterCssText : function (className) {
        var rules = isc.Canvas.getStyleRules(this.styleSheet);
        if (rules == null) return false;

        if (!isc.isAn.Array(className)) className = [className];

        // map of className to compound cssText
        var classCSS = {};

        var classList = [];
        // make sure there's a leading "."
        for (var i = 0; i < className.length; i++) {
            classList.add({ name: this.formatClassName(className[i]), cssText: "" });
        }

        // build a single cssText from all selectors for the className
        for (var i = 0; i < rules.length; i++) {
            var selectorText = rules[i].selectorText;
            if (selectorText == null) continue;

            // split the selectorText into classNames
            var classes = selectorText.trim().split(",").callMethod("trim");

            // for each of the classNames
            for (var j=0; j<classList.length; j++) {
                var name = classList[j].name;

                var filtered = classes.filter(function (el) { return el == name; });
                if (filtered.length > 0) {
                    // rule is for multiple classes, including the passed "className"
                    classList[j].cssText += rules[i].style.cssText;
                }
            }
            
        }

        var result = "";
        for (var j=0; j<classList.length; j++) {
            // split the cssText into an array
            var settings = classList[j].cssText.split(";").callMethod("trim");
            var props = {};
            
            // apply the settings in order so that, if different rules affect the same settings, 
            // the value from the latest rule will win
            for (var i=0; i<settings.length; i++) {
                if (!settings[i]) continue;
                var parts = settings[i].split(":").callMethod("trim");
                props[parts[0]] = parts[1];
            }
            if (isc.getValues(props).length == 0) continue;
            var css = classList[j].name + " { ";
            for (var key in props) {
                css += key + ":" + props[key] + "; ";
            }
            css += "} ";
            
            result += css;
        }
        return result;
    },

    //> @method StyleSheetHandler.getClassList()
    // Returns the list of CSS class-names defined in this handler's
    // +link{styleSheetHandler.styleSheet, stylesheet}.
    // @return (Array of String) the CSS declaration for the passed className
    // @visibility external
    //<
    getClassList : function () {
        if (this._classList) return this._classList;

        var rules = isc.Canvas.getStyleRules(this.styleSheet);
        if (rules == null) return false;

        var classNames = this._classList = [];

        // get a list of classes in the stylesheet
        for (var i = 0; i < rules.length; i++) {
            var selectorText = rules[i].selectorText.trim();
            if (selectorText == null) continue;
            // split the selectorText into classNames
            var classes = selectorText.split(",").callMethod("trim");
            var filtered = classes.filter(function (el) { return el && el.startsWith("."); });
            classNames.addList(filtered);
        }

        this._classList = classNames.getUniqueItems();
        return this._classList;
    },
    clearClassListCache : function () {
        delete this._classList;
    },
    hasClass : function (className) {
        className = this.formatClassName(className);
        return this.getClassList().contains(className);
    },


    // utilities

    //> @method StyleSheetHandler.loaded()
    // Has this handler's +link{styleSheetHandler.styleSheet, styleSheet} been loaded into the 
    // DOM?
    // @visibility external
    //<
    loaded : function () {
        return this.styleSheet != null;
    },
    
    formatClassName : function (className) {
        if (!className) return;
        // class rules (excluding @font-face/@media/etc) need a prefixing "."
        if (!className.startsWith("@") && !className.startsWith(".")) className = "." + className;
        return className;
    },
    refreshFramework : function () {
        // widgets like buttons (which push styles around among child-elements) may need to be 
        // notified that styles have changed
        isc.Element.cssVariablesUpdated();
    },
    
    destroy : function () {
        if (this.loaded()) this.unload();
        return this.Super("destroy");
    }
});

